<?php

use yii\helpers\Html;
use yii\helpers\Url;
use yii\widgets\ActiveForm;

$this->title = 'Передача показаний';
$this->params['breadcrumbs'][] = ['label' => 'Мои показания', 'url' => ['index']];
$this->params['breadcrumbs'][] = $this->title;
?>

<div class="meter-reading-create">
    <div style="max-width: 800px; margin: 0 auto;">
        <div class="card">
            <div class="card-header">
                <h1 class="card-title" style="text-align: center; margin: 0;"><?= Html::encode($this->title) ?></h1>
            </div>
            <div class="card-body">
                <p style="text-align: center; color: #666; margin-bottom: 20px;">Заполните форму для передачи показаний счетчиков</p>

                <?php $form = ActiveForm::begin([
                    'id' => 'meter-reading-form',
                    'options' => ['class' => 'form']
                ]); ?>

                <div style="margin-bottom: 25px;">
                    <?= $form->field($model, 'service_category_id')->dropDownList(
                        array_reduce($categories, function($carry, $category) {
                            $carry[$category->id] = $category->name . ' (' . $category->unit . ', тариф: ' . number_format($category->tariff, 2, ',', ' ') . ' ₽)';
                            return $carry;
                        }, []),
                        [
                            'class' => 'form-control',
                            'prompt' => '-- Выберите услугу --',
                            'id' => 'service-category'
                        ]
                    )->label('Коммунальная услуга') ?>
                </div>

                <div style="background: #f8f9fa; border-radius: 8px; padding: 20px; margin-bottom: 25px;">
                    <h4 style="margin-top: 0; margin-bottom: 15px; color: #2c3e50;">Информация о выбранной услуге</h4>
                    <div style="display: grid; grid-template-columns: repeat(auto-fit, minmax(200px, 1fr)); gap: 15px;">
                        <div>
                            <p style="margin: 0; color: #666; font-size: 0.9rem;">Предыдущие показания:</p>
                            <p style="margin: 0; font-weight: bold; font-size: 1.2rem;" id="previous-reading">0</p>
                        </div>
                        <div>
                            <p style="margin: 0; color: #666; font-size: 0.9rem;">Единица измерения:</p>
                            <p style="margin: 0; font-weight: bold; font-size: 1.2rem;" id="unit">-</p>
                        </div>
                        <div>
                            <p style="margin: 0; color: #666; font-size: 0.9rem;">Тариф:</p>
                            <p style="margin: 0; font-weight: bold; font-size: 1.2rem;" id="tariff">-</p>
                        </div>
                    </div>
                </div>

                <div style="display: grid; grid-template-columns: repeat(auto-fit, minmax(300px, 1fr)); gap: 20px; margin-bottom: 25px;">
                    <div>
                        <?= $form->field($model, 'previous_reading')->textInput([
                            'class' => 'form-control',
                            'type' => 'number',
                            'step' => '0.001',
                            'min' => '0',
                            'id' => 'previous-reading-input',
                            'readonly' => true,
                            'style' => 'background-color: #f8f9fa;'
                        ])->label('Предыдущие показания (автоматически)') ?>
                    </div>
                    <div>
                        <?= $form->field($model, 'current_reading')->textInput([
                            'class' => 'form-control',
                            'type' => 'number',
                            'step' => '0.001',
                            'min' => '0',
                            'id' => 'current-reading-input',
                            'placeholder' => 'Введите текущие показания'
                        ])->label('Текущие показания') ?>
                    </div>
                </div>

                <div style="background: #e8f4fd; border: 1px solid #bbdefb; border-radius: 8px; padding: 20px; margin-bottom: 25px;">
                    <h4 style="margin-top: 0; margin-bottom: 15px; color: #1565c0;">Расчет</h4>
                    <div style="display: grid; grid-template-columns: repeat(auto-fit, minmax(200px, 1fr)); gap: 15px;">
                        <div>
                            <p style="margin: 0; color: #666; font-size: 0.9rem;">Расход:</p>
                            <p style="margin: 0; font-weight: bold; font-size: 1.2rem;" id="consumption">0</p>
                        </div>
                        <div>
                            <p style="margin: 0; color: #666; font-size: 0.9rem;">Сумма к оплате:</p>
                            <p style="margin: 0; font-weight: bold; font-size: 1.4rem; color: #27ae60;" id="amount">0 ₽</p>
                        </div>
                    </div>
                </div>

                <div style="margin-top: 20px;">
                    <?= Html::submitButton('Отправить показания', [
                        'class' => 'btn btn-success',
                        'name' => 'submit-button',
                        'style' => 'width: 100%; padding: 12px; font-size: 1.1rem;'
                    ]) ?>
                </div>

                <?php ActiveForm::end(); ?>
            </div>
        </div>

        <div style="margin-top: 30px; text-align: center;">
            <?= Html::a('Вернуться к списку показаний', ['index'], ['class' => 'btn btn-link']) ?>
        </div>
    </div>
</div>

<script>
document.addEventListener('DOMContentLoaded', function() {
    const categorySelect = document.getElementById('service-category');
    const previousReadingSpan = document.getElementById('previous-reading');
    const previousReadingInput = document.getElementById('previous-reading-input');
    const currentReadingInput = document.getElementById('current-reading-input');
    const unitSpan = document.getElementById('unit');
    const tariffSpan = document.getElementById('tariff');
    const consumptionSpan = document.getElementById('consumption');
    const amountSpan = document.getElementById('amount');
    
    // Данные о категориях из PHP
    const categoriesData = <?= json_encode(array_map(function($cat) use ($lastReadings) {
        return [
            'id' => $cat->id,
            'name' => $cat->name,
            'unit' => $cat->unit,
            'tariff' => $cat->tariff,
            'lastReading' => $lastReadings[$cat->id] ?? 0
        ];
    }, $categories)) ?>;
    
    // Форматирование чисел
    function formatNumber(num, decimals = 3) {
        return parseFloat(num).toFixed(decimals).replace('.', ',').replace(/\B(?=(\d{3})+(?!\d))/g, ' ');
    }
    
    function formatCurrency(num) {
        return formatNumber(num, 2) + ' ₽';
    }
    
    // Обработчик изменения категории
    categorySelect.addEventListener('change', function() {
        const categoryId = parseInt(this.value);
        const category = categoriesData.find(c => c.id === categoryId);
        
        if (category) {
            previousReadingSpan.textContent = formatNumber(category.lastReading);
            previousReadingInput.value = category.lastReading;
            unitSpan.textContent = category.unit;
            tariffSpan.textContent = formatNumber(category.tariff, 2) + ' ₽/' + category.unit;
            
            // Обновляем расчет
            updateCalculation();
        } else {
            previousReadingSpan.textContent = '0';
            previousReadingInput.value = '0';
            unitSpan.textContent = '-';
            tariffSpan.textContent = '-';
            consumptionSpan.textContent = '0';
            amountSpan.textContent = '0 ₽';
        }
    });
    
    // Обработчик изменения текущих показаний
    currentReadingInput.addEventListener('input', updateCalculation);
    
    function updateCalculation() {
        const previous = parseFloat(previousReadingInput.value) || 0;
        const current = parseFloat(currentReadingInput.value) || 0;
        const categoryId = parseInt(categorySelect.value);
        const category = categoriesData.find(c => c.id === categoryId);
        
        if (category && current >= previous) {
            const consumption = current - previous;
            const amount = consumption * category.tariff;
            
            consumptionSpan.textContent = formatNumber(consumption);
            amountSpan.textContent = formatCurrency(amount);
        } else {
            consumptionSpan.textContent = '0';
            amountSpan.textContent = '0 ₽';
        }
    }
    
    // Проверка формы перед отправкой
    document.getElementById('meter-reading-form').addEventListener('submit', function(e) {
        const current = parseFloat(currentReadingInput.value) || 0;
        const previous = parseFloat(previousReadingInput.value) || 0;
        
        if (current <= previous) {
            e.preventDefault();
            alert('Текущие показания должны быть больше предыдущих!');
            currentReadingInput.focus();
        }
    });
});
</script>