<?php

namespace app\models;

use Yii;
use yii\db\ActiveRecord;
use yii\web\IdentityInterface;

/**
 * User model
 *
 * @property integer $id
 * @property string $full_name
 * @property string $phone
 * @property string $email
 * @property string $address
 * @property integer $residents_count
 * @property string $password_hash
 * @property string $auth_key
 * @property integer $created_at
 * @property integer $updated_at
 */
class User extends ActiveRecord implements IdentityInterface
{
    public $password;
    public $password_repeat;
    public $agree_terms;

    const SCENARIO_REGISTER = 'register';
    const SCENARIO_UPDATE = 'update';
    const SCENARIO_CHANGE_PASSWORD = 'changePassword';

    /**
     * {@inheritdoc}
     */
    public static function tableName()
    {
        return '{{%user}}';
    }

    /**
     * {@inheritdoc}
     */
    public function rules()
    {
        return [
            // Регистрация
            [['full_name', 'phone', 'email', 'address', 'residents_count', 'password', 'password_repeat', 'agree_terms'], 'required', 'on' => self::SCENARIO_REGISTER],
            
            // Обновление профиля
            [['full_name', 'phone', 'email', 'address', 'residents_count'], 'required', 'on' => self::SCENARIO_UPDATE],
            
            // Смена пароля
            [['password', 'password_repeat'], 'required', 'on' => self::SCENARIO_CHANGE_PASSWORD],
            
            // Валидация ФИО
            ['full_name', 'match', 'pattern' => '/^[а-яА-ЯёЁ\s\-]+$/u', 'message' => 'ФИО должно содержать только русские буквы, пробелы и дефисы'],
            
            // Валидация телефона
            ['phone', 'match', 'pattern' => '/^8\(\d{3}\)\d{3}-\d{2}-\d{2}$/', 'message' => 'Телефон должен быть в формате 8(XXX)XXX-XX-XX'],
            ['phone', 'unique', 'message' => 'Этот телефон уже зарегистрирован'],
            
            // Валидация email
            ['email', 'email'],
            ['email', 'unique', 'message' => 'Этот email уже зарегистрирован'],
            
            // Валидация количества проживающих
            ['residents_count', 'integer', 'min' => 1],
            
            // Валидация пароля
            ['password', 'string', 'min' => 10, 'message' => 'Пароль должен содержать минимум 10 символов'],
            ['password_repeat', 'compare', 'compareAttribute' => 'password', 'message' => 'Пароли не совпадают'],
            
            // Согласие с правилами
            ['agree_terms', 'compare', 'compareValue' => true, 'message' => 'Вы должны согласиться с правилами', 'on' => self::SCENARIO_REGISTER],
            
            // Безопасность
            [['password', 'password_repeat', 'agree_terms'], 'safe'],
        ];
    }public function scenarios()
{
    $scenarios = parent::scenarios();
    $scenarios[self::SCENARIO_REGISTER] = ['full_name', 'phone', 'email', 'address', 'residents_count', 'password', 'password_repeat', 'agree_terms'];
    $scenarios[self::SCENARIO_UPDATE] = ['full_name', 'phone', 'email', 'address', 'residents_count'];
    $scenarios[self::SCENARIO_CHANGE_PASSWORD] = ['password', 'password_repeat'];
    return $scenarios;
}

    /**
     * {@inheritdoc}
     */
    public function attributeLabels()
    {
        return [
            'id' => 'ID',
            'full_name' => 'ФИО',
            'phone' => 'Телефон',
            'email' => 'Email',
            'address' => 'Адрес проживания',
            'residents_count' => 'Количество проживающих',
            'password' => 'Пароль',
            'password_repeat' => 'Повтор пароля',
            'agree_terms' => 'Согласие с правилами регистрации',
            'created_at' => 'Дата регистрации',
            'updated_at' => 'Дата обновления',
        ];
    }

    /**
     * {@inheritdoc}
     */
    public function beforeSave($insert)
    {
        if (parent::beforeSave($insert)) {
            if ($insert) {
                $this->created_at = time();
                $this->auth_key = Yii::$app->security->generateRandomString();
            }
            
            // Если указан пароль - хэшируем его
            if ($this->password) {
                $this->password_hash = Yii::$app->security->generatePasswordHash($this->password);
            }
            
            $this->updated_at = time();
            return true;
        }
        return false;
    }

    /**
     * {@inheritdoc}
     */
    public function afterSave($insert, $changedAttributes)
    {
        parent::afterSave($insert, $changedAttributes);
        
        // Очищаем пароль после сохранения
        $this->password = null;
        $this->password_repeat = null;
    }

    /**
     * IdentityInterface methods
     */
    
    /**
     * {@inheritdoc}
     */
    public static function findIdentity($id)
    {
        return static::findOne($id);
    }

    /**
     * {@inheritdoc}
     */
    public static function findIdentityByAccessToken($token, $type = null)
    {
        return null; // Не используется
    }

    /**
     * {@inheritdoc}
     */
    public function getId()
    {
        return $this->id;
    }

    /**
     * {@inheritdoc}
     */
    public function getAuthKey()
    {
        return $this->auth_key;
    }

    /**
     * {@inheritdoc}
     */
    public function validateAuthKey($authKey)
    {
        return $this->auth_key === $authKey;
    }

    /**
     * Найти пользователя по телефону
     */
    public static function findByPhone($phone)
    {
        return static::findOne(['phone' => $phone]);
    }

    /**
     * Валидация пароля
     */
    public function validatePassword($password)
    {
        return Yii::$app->security->validatePassword($password, $this->password_hash);
    }

    /**
     * Генерирует хэш пароля
     */
    public function setPassword($password)
    {
        $this->password_hash = Yii::$app->security->generatePasswordHash($password);
    }

    /**
     * Генерирует ключ авторизации
     */
    public function generateAuthKey()
    {
        $this->auth_key = Yii::$app->security->generateRandomString();
    }
}