<?php

namespace app\models;

use Yii;
use yii\db\ActiveRecord;

/**
 * This is the model class for table "news".
 *
 * @property int $id
 * @property string $title
 * @property string $content
 * @property string $publish_date
 * @property int $is_active
 * @property int $created_at
 * @property int $updated_at
 */
class News extends ActiveRecord
{
    /**
     * {@inheritdoc}
     */
    public static function tableName()
    {
        return '{{%news}}';
    }

    /**
     * {@inheritdoc}
     */
    public function rules()
    {
        return [
            [['title', 'content', 'publish_date'], 'required'],
            [['content'], 'string'],
            [['publish_date'], 'safe'],
            [['is_active', 'created_at', 'updated_at'], 'integer'],
            [['title'], 'string', 'max' => 255],
            [['is_active'], 'default', 'value' => 1],
        ];
    }

    /**
     * {@inheritdoc}
     */
    public function attributeLabels()
    {
        return [
            'id' => 'ID',
            'title' => 'Заголовок',
            'content' => 'Содержание',
            'publish_date' => 'Дата публикации',
            'is_active' => 'Активна',
            'created_at' => 'Дата создания',
            'updated_at' => 'Дата обновления',
        ];
    }

    /**
     * {@inheritdoc}
     */
    public function beforeSave($insert)
    {
        if (parent::beforeSave($insert)) {
            if ($insert) {
                $this->created_at = time();
            }
            $this->updated_at = time();
            return true;
        }
        return false;
    }

    /**
     * Получает активные новости
     */
    public static function getActiveNews($limit = 5)
    {
        return self::find()
            ->where(['is_active' => 1])
            ->andWhere(['<=', 'publish_date', date('Y-m-d')])
            ->orderBy(['publish_date' => SORT_DESC, 'id' => SORT_DESC])
            ->limit($limit)
            ->all();
    }

    /**
     * Получает все новости
     */
    public static function getAllNews()
    {
        return self::find()
            ->orderBy(['publish_date' => SORT_DESC, 'id' => SORT_DESC])
            ->all();
    }

    /**
     * Форматирует дату публикации
     */
    public function getFormattedPublishDate()
    {
        return Yii::$app->formatter->asDate($this->publish_date, 'long');
    }

    /**
     * Обрезает контент для превью
     */
    public function getShortContent($length = 200)
    {
        $content = strip_tags($this->content);
        if (strlen($content) > $length) {
            return substr($content, 0, $length) . '...';
        }
        return $content;
    }
}